/*
 * Copyright (c) 1998
 * Silicon Graphics Computer Systems, Inc.
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Silicon Graphics makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 */ 



#include <fstream>
#ifdef __unix
#  include <sys/types.h>
#  include <sys/stat.h>         // For stat
#  include <sys/mman.h>         // For mmap
#  include <fcntl.h>            // For open, close
#include <unistd.h>

// For Windows, we (temporarily) use functions that emulate Unix system
// calls.  We should replace this scheme by native Win32 system calls.
#elif _MSC_VER
#  define UNIX_EMULATION
#  include <io.h>
#  include <fcntl.h>
#  include <sys/stat.h>
#endif /* __unix */

#if defined(__unix)
#  define OPEN   open
#  define CLOSE  close
#  define READ   read
#  define WRITE  write
#  define STAT   stat
#  define FSTAT  fstat
#  define FILENO fileno
#  define DEFAULT_PERMISSIONS \
          (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)
#  define IS_REGULAR(mod) S_ISREG(mod)

   // All version of Unix have mmap and lseek system calls.  Some also have
   // longer versions of those system calls to accommodate 64-bit offsets.
#  ifdef __sgi /* IRIX */
#    define LSEEK lseek64
#    define MMAP  mmap64
#  else
#    define LSEEK lseek
#    define MMAP  mmap
#  endif

#elif defined(UNIX_EMULATION)
#  define OPEN   _open
#  define CLOSE  _close
#  define READ   _read
#  define WRITE  _write
#  define STAT   _stat
#  define FSTAT  _fstat
#  define FILENO fileno
#  define DEFAULT_PERMISSIONS (_S_IREAD | _S_IWRITE)
#  define IS_REGULAR(mod) ((mod & _S_IFREG) != 0)
#  define LSEEK _lseek
   // Unix emulation doesn't provide mmap.  That's one reason to 
   // use native Win32 system calls instead.

# ifndef O_RDONLY
#   define O_RDONLY _O_RDONLY
#   define O_WRONLY _O_WRONLY
#   define O_RDWR   _O_RDWR
#   define O_APPEND _O_APPEND
#   define O_CREAT  _O_CREAT
#   define O_TRUNC  _O_TRUNC
#   define O_TEXT   _O_TEXT
#   define O_BINARY _O_BINARY
# endif

#endif

// Helper functions for _Filebuf_base.

namespace SGI {

bool __is_regular_file(int fd) {
#if defined(__unix) || defined(UNIX_EMULATION)
  struct STAT buf;
  return FSTAT(fd, &buf) == 0 && IS_REGULAR(buf.st_mode);
#else
  return false;
#endif /* unix or unix emulation */
}

// Number of characters in the file.  
std::streamoff __file_size(int fd) {
#if defined(__unix) || defined(UNIX_EMULATION)
  struct STAT buf;
  if(FSTAT(fd, &buf) == 0 && IS_REGULAR(buf.st_mode))
    return buf.st_size > 0 ? buf.st_size : 0;
  else
    return 0;
#else
  return 0;
#endif /* unix or unix emulation */
}

// Return the number of characters in f that follow the end of the 
// internal buffer.  Returns 0 if no estimate is possible.
std::streamoff __remaining_characters(FILE* f)
{
  int fd = -1;
#if defined(__unix) || defined(UNIX_EMULATION)
  fd = FILENO(f);
#endif

  std::streamoff size = SGI::__file_size(fd);
  long pos = ftell(f);

  return pos >= 0 && size > pos ? size - pos : 0;
}

} // Close namespace SGI

__STL_BEGIN_NAMESPACE

_Filebuf_base::_Filebuf_base()
  : _M_file_id(-1),
    _M_openmode(0),
    _M_page_size(512),          // Just a guess
    _M_is_open(false),
    _M_should_close(false)
{
#ifdef __unix
  _M_page_size = sysconf(_SC_PAGESIZE);
#endif
}

bool _Filebuf_base::_M_open(const char* name, ios_base::openmode openmode,
                            long permission)
{
  if (_M_is_open)
    return false;

#if defined(__unix) || defined(UNIX_EMULATION)
  int flags = 0;

  switch(openmode & (~ios_base::ate & ~ios_base::binary)) {
  case ios_base::out:
  case ios_base::out | ios_base::trunc:
    flags = O_WRONLY | O_CREAT | O_TRUNC;
    break;
  case ios_base::out | ios_base::app:
    flags = O_WRONLY | O_CREAT | O_APPEND;
    break;
  case ios_base::in:
    flags = O_RDONLY;
    permission = 0;             // Irrelevant unless we're writing.
    break;
  case ios_base::in | ios_base::out:
    flags = O_RDWR;
    break;
  case ios_base::in | ios_base::out | ios_base::trunc:
    flags = O_RDWR | O_CREAT | O_TRUNC;
    break;
  default:                      // The above are the only combinations of 
    return false;               // flags allowed by the C++ standard.
  }

  // Unix makes no distinction between text and binary mode, but Windows does.
# if defined(UNIX_EMULATION)
  flags |= openmode & ios_base::binary ? _O_BINARY : _O_TEXT;
# endif

  int fileno = OPEN(name, flags, permission);
  if (fileno < 0)
    return false;
  else {
    _M_is_open = true;
    _M_file_id = fileno;

    if (openmode & ios_base::ate)
      if (LSEEK(fileno, 0, SEEK_END) == -1)
        _M_is_open = false;
  }

#endif /* __unix */

  _M_should_close = _M_is_open;
  _M_openmode = openmode;
  if (_M_is_open)
    _M_regular_file = SGI::__is_regular_file(_M_file_id);

  return _M_is_open;
}

bool _Filebuf_base::_M_open(const char* name, ios_base::openmode openmode)
{
  // This doesn't really grant everyone in the world read/write
  // access.  On Unix, file-creation system calls always clear 
  // bits that are set in the umask from the permissions flag.
  return this->_M_open(name, openmode, DEFAULT_PERMISSIONS);
}

// Associate the filebuf with a file descriptor pointing to an already-
// open file.  Mode is set to be consistent with the way that the file
// was opened.
bool _Filebuf_base::_M_open(int fileno) {

#if defined (__unix)
  if (_M_is_open || fileno < 0)
    return false;

  // How do we do this with Unix emulation?  Microsoft's emulation library
  // doesn't provide fcntl.
  int mode = fcntl(fileno, F_GETFL);
  if (mode == -1)
    return false;

  switch(mode & O_ACCMODE) {
  case O_RDONLY:
    _M_openmode = ios_base::in; break;
  case O_WRONLY:
    _M_openmode = ios_base::out; break;
  case O_RDWR:
    _M_openmode = ios_base::in | ios_base::out; break;
  default:
    return false;
  }

  if (mode & O_APPEND)
    _M_openmode |= ios_base::app;

  _M_is_open = true;
  _M_file_id = fileno;
  _M_should_close = false;
  _M_regular_file = SGI::__is_regular_file(_M_file_id);

  return true;

#else
  return false;
#endif
}

bool _Filebuf_base::_M_close() {
  if (!_M_is_open)
    return false;

  bool ok = false;

  if (!_M_should_close)
    ok = true;
  else {
#   if defined(__unix) || defined(UNIX_EMULATION)
    ok = CLOSE(_M_file_id) == 0;
#   endif /* __unix */
  }

  _M_is_open = _M_should_close = false;
  _M_openmode = 0;
  return ok;
}

// Read up to n characters into a buffer.  Return value is number of
// characters read.
ptrdiff_t _Filebuf_base::_M_read(char* buf, ptrdiff_t n) {
# if defined( __unix ) || defined(UNIX_EMULATION) 
  return READ(_M_file_id, buf, n);
# endif /* __unix */
}

// Write n characters from a buffer.  Return value: true if we managed
// to write the entire buffer, false if we didn't.  
bool _Filebuf_base::_M_write(char* buf, ptrdiff_t n) {
  while (true) {
    ptrdiff_t written = 0;
#   if defined(__unix) || defined(UNIX_EMULATION)
    written = WRITE(_M_file_id, buf, n);
#   endif /* __unix */ 
    if (n == written)
      return true;
    else if (written > 0 && written < n) {
      n -= written;
      buf += written;
    }
    else
      return false;
  }
}

// Wrapper for lseek or the like.
streamoff _Filebuf_base::_M_seek(streamoff offset, ios_base::seekdir dir)
{
  streamoff result = -1;

# if defined(__unix) || defined(UNIX_EMULATION)

  int whence;

  switch(dir) {
  case ios_base::beg:
    if (offset < 0 || offset > _M_file_size())
      return -1;
    whence = SEEK_SET;
    break;
  case ios_base::cur:
    whence = SEEK_CUR;
    break;
  case ios_base::end:
    if (offset > 0 || -offset > _M_file_size())
      return -1;
    whence = SEEK_END;
    break;
  default:
    return streamoff(-1);
  }

  result = LSEEK(_M_file_id, offset, whence);

# endif

  return result;
}

// Return the size of the file.  In Unix, this is a wrapper for stat.
// Returns zero if the size cannot be determined or is ill-defined.
streamoff _Filebuf_base::_M_file_size()
{
  return SGI::__file_size(_M_file_id);
}

// Attempts to memory-map len bytes of the current file, starting
// at position offset.  Precondition: offset is a multiple of the
// page size.  Postcondition: return value is a null pointer if the
// memory mapping failed.  Otherwise the return value is a pointer to 
// the memory-mapped file and the file position is set to offset.
void* _Filebuf_base::_M_mmap(streamoff offset, streamoff len) {
#ifdef __unix
  void* base = MMAP(0, len, PROT_READ, MAP_PRIVATE, _M_file_id, offset);
  if (base != MAP_FAILED) {
    if (LSEEK(_M_file_id, offset + len, SEEK_SET) < 0) {
      this->_M_unmap(base, len);
      base = 0;
    }
  }
  return base;
#else
  return 0;
#endif
}

void _Filebuf_base::_M_unmap(void* base, streamoff len) {
#ifdef __unix
  munmap((caddr_t) base, len);
#endif
}

// In Unix, writing n characters always bumps the file position by n.
// In Windows text mode, however, it bumps the file position by n + m,
// where m is the number of newlines in the range.  That's because an
// internal \n corresponds to an external two-character sequence.
streamoff _Filebuf_base::_M_get_offset(char* first, char* last)
{
#ifdef __unix
  return last - first;
#elif _MSC_VER
  int n;
  count(first, last, '\n', n);
  return (_M_openmode & ios_base::binary)
    ? last - first
    : (last - first) + n;
#endif  
}

// In Unix there is no distinction between text and binary mode.  In Windows
// the difference is that text mode converts \n to an external two-character
// sequence.
bool _Filebuf_base::_M_in_binary_mode() const 
{
#ifdef __unix
  return true;
#else
  return _M_openmode & ios_base::binary;
#endif
}

// Specialization of underflow: if the character type is char, maybe
// we can use mmap instead of read.
template<>
class _Underflow< basic_filebuf<char, char_traits<char> > > {
public:
  typedef basic_filebuf<char, char_traits<char> >::int_type int_type;
  typedef char_traits<char> traits_type;
  
  static inline int_type doit(basic_filebuf<char, traits_type >* __this) 
  {
    
    if (!__this->_M_in_input_mode) {
      if (!__this->_M_switch_to_input_mode())
	return traits_type::eof();
    }
    
    else if (__this->_M_in_putback_mode) {
      __this->_M_exit_putback_mode();
      if (__this->gptr() != __this->egptr()) {
	int_type __c = traits_type::to_int_type(*__this->gptr());
	return __c;
      }
    }
    
    // If it's a disk file, and if the internal and external character
    // sequences are guaranteed to be identical, then try to use memory
    // mapped I/O.  Otherwise, revert to ordinary read.
    if (__this->_M_regular_file 
	&& __this->_M_always_noconv
	&& __this->_M_in_binary_mode()) {
      // If we've mmapped part of the file already, then unmap it.
      if (__this->_M_mmap_base)
	__this->_M_unmap(__this->_M_mmap_base, __this->_M_mmap_len);
      
      // Determine the position where we start mapping.  It has to be
      // a multiple of the page size.
      streamoff __cur = __this->_M_seek(0, ios_base::cur);
      streamoff __size = __this->_M_file_size();
      if (__size > 0 && __cur >= 0 && __cur < __size) {
	streamoff __offset    = (__cur / __this->_M_page_size)
	  * __this->_M_page_size;
	streamoff __remainder = __cur - __offset;
	
	__this->_M_mmap_len = __size - __offset;
	if ((__this->_M_mmap_base =
	     __this->_M_mmap(__offset, __this->_M_mmap_len)) != 0) {
	  __this->setg((char*) __this->_M_mmap_base,
		       (char*) __this->_M_mmap_base + __remainder,
		       (char*) __this->_M_mmap_base + __this->_M_mmap_len);
	  return traits_type::to_int_type(*__this->gptr());
	}
      }
    }
    
    return __this->_M_underflow_aux();
  }
};

//----------------------------------------------------------------------
// Force instantiation of filebuf and fstream classes.

template class basic_filebuf<char>;
template class basic_ifstream<char>;
template class basic_ofstream<char>;
template class basic_fstream<char>;

#ifdef INSTANTIATE_WIDE_STREAMS
template class basic_filebuf<wchar_t>;
template class basic_ifstream<wchar_t>;
template class basic_ofstream<wchar_t>;
template class basic_fstream<wchar_t>;
#endif /* INSTANTIATE_WIDE_STREAMS */


__STL_END_NAMESPACE

// Local Variables:
// mode:C++
// End:
